# -*- coding: utf-8 -*-
from importlib import import_module
from typing import Type

import django
from django.conf import settings
from django.contrib.auth import BACKEND_SESSION_KEY, load_backend, \
    HASH_SESSION_KEY, SESSION_KEY
from django.db import models
from django.utils.crypto import constant_time_compare

from rpc_framework.context import RPCViewContext
from rpc_framework.models import GaiaUser

from django.contrib.sessions.models import Session as DjangoSession
from django.contrib.sessions.backends.db import SessionStore as DjangoSessionStore

# class Session(DjangoSession):
#     objects = models.Manager().db_manager(using='gaia')


# class SessionStore(DjangoSessionStore):
#     @classmethod
#     def get_model_class(cls):
        # Avoids a circular import and allows importing SessionStore when
        # django.contrib.sessions is not in INSTALLED_APPS.
        # return Session


def get_user(context):
    """
    Return the user model instance associated with the given request session.
    If no user is retrieved, return an instance of `AnonymousUser`.
    """
    if getattr(settings, 'AUTH_USER_USING_GAIA_DB', False):
        return get_user_using_gaia_db(context)
    else:
        return get_user_using_passport_service(context)


def get_session_store_class() -> Type[django.contrib.sessions.backends.db.SessionStore]:
    # django SessionMiddleware
    engine = import_module(settings.SESSION_ENGINE)
    return engine.SessionStore


def get_user_using_gaia_db(context: RPCViewContext):
    from django.contrib.auth.models import AnonymousUser
    user = None
    try:
        session_store_class = get_session_store_class()
        session = session_store_class(context._context.session_id)

        user_id = GaiaUser._meta.pk.to_python(session[SESSION_KEY])
        backend_path = session[BACKEND_SESSION_KEY]
    except KeyError:
        pass
    else:
        if backend_path in settings.AUTHENTICATION_BACKENDS:
            backend = load_backend(backend_path)
            user = backend.get_user(user_id)
            # Verify the session
            if hasattr(user, 'get_session_auth_hash'):
                session_hash = session.get(HASH_SESSION_KEY)
                session_hash_verified = session_hash and constant_time_compare(
                    session_hash,
                    user.get_session_auth_hash()
                )
                if not session_hash_verified:
                    session.flush()
                    user = None

    return user or AnonymousUser()


def get_user_using_passport_service(context):
    pass
