# -*- coding: utf-8 -*-
from abc import ABCMeta, abstractmethod
from typing import Sequence, Optional, Union, Dict, List

from gm_rpcd.internals.exceptions import RPCDFaultException
from pydantic import ValidationError, create_model
from pydantic.error_wrappers import ErrorList, ErrorWrapper

from extension.types import Error


def _check_methods(C, *methods):
    mro = C.__mro__
    for method in methods:
        for B in mro:
            if method in B.__dict__:
                if B.__dict__[method] is None:
                    return NotImplemented
                break
        else:
            return NotImplemented
    return True


class RPCViewBaseException(Exception, metaclass=ABCMeta):
    code: int
    message: str
    data: Optional[Union[Dict, List]]

    __slots__ = ()

    @classmethod
    def __subclasshook__(cls, C):
        if cls is RPCViewBaseException:
            return _check_methods(C, "as_rpc_view_exception")
        return NotImplemented


RequestErrorModel = create_model("Request")

class CalleeTypeHintValidationError(RPCViewBaseException):
    code = Error.PARAMS_INVALID
    message = ''

    @classmethod
    def from_error_wrapper_list(cls, errors:List[ErrorWrapper]):
        validation_error = ValidationError(errors, RequestErrorModel)
        self = cls()
        self.message = validation_error.json()
        return self


class AuthenticationFailed(RPCViewBaseException):
    code = Error.AUTH_FAILED
    message = Error.getDesc(Error.AUTH_FAILED)

class NotFoundException(RPCViewBaseException):
    code = 1404
    message = "Not Found"
    # default_message = "Not Found"


