# -*- coding: utf-8 -*-
import logging
import os
import uuid

from django.conf import settings

from gm_rpcd.internals.configuration.model import Config, environ, from_property
from gm_rpcd.internals.configuration.model_base import literal, xml_text, xml_text_list, \
    DefaultConfigPropertyWrapper, EnvironmentConfigProperty

DISPATCHER = None

GM_RPCD_APP_CONF_PATH_KEY = 'GM_RPCD_APP_CONF_PATH'
GM_RPCD_DEVELOP_CONF_PATH_KEY = 'GM_RPCD_DEVELOP_CONF_PATH'
GM_RPCD_DEPLOY_CONF_PATH_KEY = 'GM_RPCD_DEPLOY_CONF_PATH'
GM_RPCD_MODE_KEY = 'GM_RPCD_MODE'


class DjangoRpcdConfig(Config):
    '''因为 最终会从实例的wrapped属性解析出配置， 而wrapped实际上读取的是Config实例
    的`.__class__.__dict__.`,因此父类中的类成员无法继承, 所以父类中的类成员都要重写'''
    process_unique_id = literal(uuid.uuid4().hex)

    app_conf_path = DefaultConfigPropertyWrapper(gm_rpcd_app_path) \
        if (gm_rpcd_app_path := getattr(settings, GM_RPCD_APP_CONF_PATH_KEY, None)) \
        else environ('GM_RPCD_APP_CONF_PATH').default(os.path.abspath('app_conf.xml'))

    develop_conf_path = DefaultConfigPropertyWrapper(gm_develop_conf_path) \
            if (gm_develop_conf_path := getattr(settings, GM_RPCD_DEVELOP_CONF_PATH_KEY, None)) \
            else environ('GM_RPCD_DEVELOP_CONF_PATH').default( os.path.abspath('.gm_rpcd.develop_conf.xml'))

    deploy_conf_path = EnvironmentConfigProperty(gm_deploy_conf_path) \
            if (gm_deploy_conf_path := getattr(settings, GM_RPCD_DEPLOY_CONF_PATH_KEY, None)) \
            else environ('GM_RPCD_DEPLOY_CONF_PATH')

    # mode is instance of  `ConfigProperty('develop')` or `ConfigProperty('deploy')`
    mode = EnvironmentConfigProperty(gm_rpcd_mode) \
            if (gm_rpcd_mode := getattr(settings, GM_RPCD_MODE_KEY, None)) \
            else environ('GM_RPCD_MODE')
    is_develop_mode, is_deploy_mode  = mode.eq('develop'), mode.eq('deploy')

    application_name = xml_text('app', 'application_name')
    service_list = xml_text_list('app', 'service_list').default_func(list)
    initializer_list = xml_text_list('app', 'initializer_list').default_func(list)
    statuses_func = xml_text('app', 'statuses')

    # give app a chance for using customized request info extractor, if not
    # set, use default one GMRpcdRequestInfoExtractor
    request_info_extractor_cls = xml_text('app', 'request_info_extractor').default(None)

    log_dir_for_develop = xml_text('develop', 'log_dir').\
        default(os.path.abspath('.gm_rpcd.log'))
    log_dir_for_deploy = xml_text('deploy', 'log_dir')

    sentry_dsn_for_deploy = xml_text('deploy', 'sentry_dsn')

    log_encoding = literal('utf-8')

    @from_property
    @property
    def log_dir(self):
        if self.is_develop_mode:
            return self.log_dir_for_develop
        if self.is_deploy_mode:
            return self.log_dir_for_deploy
        raise AttributeError('invalid mode')

    @from_property
    @property
    def local_request_log_path(self):
        return os.path.join(self.log_dir, 'gm_rpcd_request.log')

    @from_property
    @property
    def local_request_error_log_path(self):
        return os.path.join(self.log_dir, 'gm_rpcd_request_error.log')


def setup_rpcd():
    logger = logging.getLogger('django.start')
    logger.info('****** setup rpcd ******')
    # call `initialize_app` of rpcd
    config_wrapper = DjangoRpcdConfig().wrapped
    config_wrapper.is_deploy_mode = True
    config_wrapper.freeze()

    ######################## PATCH #############################
    from gm_rpcd.internals.configuration import model
    model.__dict__['config'] = config_wrapper

    from gm_rpcd.internals import configuration
    configuration.__dict__['config'] = config_wrapper
    ####################### end PATCH ##########################

    from gm_rpcd.internals.initializations import initialize
    # expose to module scope to make dispatcher singleton
    global DISPATCHER
    if not DISPATCHER:
        DISPATCHER = initialize().dispatcher

    # avoid call `setup_rpcd` repeatly
    global setup_rpcd
    del setup_rpcd


setup_rpcd()


__all__ = [
    'DISPATCHER',
    'views'
]