package com.gmei.data.ctr.operator;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.gmei.data.ctr.bean.CtrEstimateTagEtl;
import com.gmei.data.ctr.bean.DeviceCurrentEstimateTagTmp;
import com.gmei.data.ctr.sink.CtrEstimateTagMysqlSink;
import com.gmei.data.ctr.source.ZhengxingMysqlAsyncSource;
import com.gmei.data.ctr.source.TidbMysqlAsyncSource;
import com.gmei.data.ctr.utils.DateUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.api.common.functions.FilterFunction;
import org.apache.flink.api.common.functions.MapFunction;
import org.apache.flink.streaming.api.datastream.AsyncDataStream;
import org.apache.flink.streaming.api.datastream.DataStream;
import org.apache.flink.streaming.api.datastream.SingleOutputStreamOperator;

import java.util.concurrent.TimeUnit;

/**
 * @ClassName CtrEstimateTagOperator
 * @Description: CTR特征预估标签表
 * @Author zhaojianwei
 * @Date 2020/4/01
 * @Version V1.0
 **/
public class CtrEstimateTagOperator implements BaseOperator{
    private DataStream dataStream;
    private String outJdbcUrl;
    private int maxRetry;
    private long retryInteral;
    private int parallelism;

    public CtrEstimateTagOperator(DataStream dataStream, String outJdbcUrl, int maxRetry, long retryInteral, int parallelism) {
        this.dataStream = dataStream;
        this.outJdbcUrl = outJdbcUrl;
        this.maxRetry = maxRetry;
        this.retryInteral = retryInteral;
        this.parallelism = parallelism;
    }

    @Override
    public void run() {
        SingleOutputStreamOperator map = dataStream
                .filter(new FilterFunction<String>() {
                    @Override
                    public boolean filter(String value) throws Exception {
                        return JSON.isValid(value);
                    }
                })
                .map(new MapFunction<String, JSONObject>() {
                    @Override
                    public JSONObject map(String value) throws Exception {
                        return JSON.parseObject(value);
                    }
                })
                .filter(new FilterFunction<JSONObject>() {
                    @Override
                    public boolean filter(JSONObject jsonObject) throws Exception {
                        Double gmNginxTimestamp = jsonObject.getDouble("gm_nginx_timestamp");
                        if (null != gmNginxTimestamp) {
                            long gmNginxTimestampLong = Math.round(gmNginxTimestamp * 1000);
                            String currentDateStr = DateUtils.getCurrentDateStr();
                            long currentDateBegin = DateUtils.getTimestampByDateStr(currentDateStr + " 00:00:00");
                            long currentDateend = DateUtils.getTimestampByDateStr(currentDateStr + " 23:59:59");
                            if (gmNginxTimestampLong >= currentDateBegin && gmNginxTimestampLong <= currentDateend) {
                                String type = jsonObject.getString("type");
                                JSONObject paramsObject = jsonObject.getJSONObject("params");
                                JSONObject deviceObject = jsonObject.getJSONObject("device");
                                if (null != paramsObject && null != deviceObject && StringUtils.isNotBlank(type)) {
                                    String deviceId = deviceObject.getString("device_id");
                                    String idfv = deviceObject.getString("idfv");
                                    String clId = "";
                                    if (StringUtils.isBlank(deviceId) && StringUtils.isNotBlank(idfv)) {
                                        clId = idfv;
                                    } else {
                                        clId = deviceId;
                                    }
                                    if (StringUtils.isNotBlank(clId)) {
                                        String cardContentType = paramsObject.getString("card_content_type");
                                        Long cardId = paramsObject.getLong("card_id");
                                        if (null != cardContentType && null != cardId) {
                                            if ("service".equals(cardContentType) || "diary".equals(cardContentType) ||
                                                    "tractate".equals(cardContentType) || "answer".equals(cardContentType)) {
                                                return true;
                                            }
                                        }
                                        if(("do_serach".equals(type) || "search_reult_click_search".equals(type)) && StringUtils.isNotBlank(paramsObject.getString("query")) ){
                                            return true;
                                        }
                                        if("on_click_card".equals(type) && null != paramsObject.getString("card_type") && "search_word".equals(paramsObject.getString("card_type"))
                                            && StringUtils.isNotBlank(paramsObject.getString("card_name"))){
                                            return true;
                                        }
                                    }
                                }
                            }
                        }
                        return false;
                    }
                })
                .map(new MapFunction<JSONObject, CtrEstimateTagEtl>() {
                    @Override
                    public CtrEstimateTagEtl map(JSONObject jsonObject) throws Exception {
                        String type = jsonObject.getString("type");
                        JSONObject paramsObject = jsonObject.getJSONObject("params");
                        JSONObject deviceObject = jsonObject.getJSONObject("device");
                        CtrEstimateTagEtl ctrEstimateTagEtl = new CtrEstimateTagEtl();
                        if (null != paramsObject && null != deviceObject && StringUtils.isNotBlank(type)) {
                            String deviceId = deviceObject.getString("device_id");
                            String idfv = deviceObject.getString("idfv");
                            String clId = "";
                            if (StringUtils.isBlank(deviceId) && StringUtils.isNotBlank(idfv)) {
                                clId = idfv;
                            } else {
                                clId = deviceId;
                            }
                            if (StringUtils.isNotBlank(clId)) {
                                String cardContentType = paramsObject.getString("card_content_type");
                                Long cardId = paramsObject.getLong("card_id");
                                ctrEstimateTagEtl.setDeviceId(deviceId);
                                if (null != cardContentType && null != cardId) {
                                    ctrEstimateTagEtl.setCardId(cardId);
                                    ctrEstimateTagEtl.setCardContentType(cardContentType);
                                    if ("service".equals(cardContentType) ) {
                                        ctrEstimateTagEtl.setType("commodity");
                                    }
                                    if("diary".equals(cardContentType) || "tractate".equals(cardContentType) || "answer".equals(cardContentType)){
                                        ctrEstimateTagEtl.setType("content");
                                    }
                                }
                                if(("do_serach".equals(type) || "search_reult_click_search".equals(type)) && StringUtils.isNotBlank(paramsObject.getString("query"))){
                                    ctrEstimateTagEtl.setType("search");
                                    ctrEstimateTagEtl.setKeyWord(paramsObject.getString("query"));
                                }
                                if ("on_click_card".equals(type) && null != paramsObject.getString("card_type") && "search_word".equals(paramsObject.getString("card_type"))
                                        && StringUtils.isNotBlank(paramsObject.getString("card_name"))){
                                    ctrEstimateTagEtl.setType("search");
                                    ctrEstimateTagEtl.setKeyWord(paramsObject.getString("card_name"));
                                }
                            }
                        }
                        return ctrEstimateTagEtl;
                    }
                });
        DataStream<DeviceCurrentEstimateTagTmp> tidbAsyncDataStream = AsyncDataStream
                .unorderedWait(map, new TidbMysqlAsyncSource(), 1, TimeUnit.MINUTES, 1000)
                .uid("tidbAsyncDataStream")
                .setParallelism(parallelism);
        DataStream<DeviceCurrentEstimateTagTmp> zhengxingAsyncDataStream = AsyncDataStream
                .unorderedWait(map, new ZhengxingMysqlAsyncSource(), 1, TimeUnit.MINUTES, 1000)
                .uid("zhengxingAsyncDataStream")
                .setParallelism(parallelism);
        DataStream<DeviceCurrentEstimateTagTmp> asyncDataStream = tidbAsyncDataStream.union(zhengxingAsyncDataStream);
        asyncDataStream
        .addSink(new CtrEstimateTagMysqlSink(outJdbcUrl, maxRetry, retryInteral))
        .setParallelism(parallelism);
    }
}
