package com.gmei.data.ctr.operator;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.gmei.data.ctr.bean.CtrEstimateClkEtl;
import com.gmei.data.ctr.bean.DeviceCurrentEstimateClk;
import com.gmei.data.ctr.sink.CtrEstimateClkMysqlSink;
import com.gmei.data.ctr.utils.DateUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.api.common.functions.FilterFunction;
import org.apache.flink.api.common.functions.MapFunction;
import org.apache.flink.api.java.functions.KeySelector;
import org.apache.flink.streaming.api.datastream.DataStream;
import org.apache.flink.streaming.api.functions.windowing.ProcessWindowFunction;
import org.apache.flink.streaming.api.windowing.time.Time;
import org.apache.flink.streaming.api.windowing.windows.TimeWindow;
import org.apache.flink.util.Collector;

import java.util.Arrays;
import java.util.Date;

/**
 * @ClassName CtrEstimateClkOperator
 * @Description: CTR特征预估点击量表
 * @Author zhaojianwei
 * @Date 2020/3/31
 * @Version V1.0
 **/
public class CtrEstimateClkOperator implements BaseOperator{
    private DataStream dataStream;
    private String outJdbcUrl;
    private int maxRetry;
    private long retryInteral;
    private int parallelism;

    public CtrEstimateClkOperator(DataStream dataStream, String outJdbcUrl, int maxRetry, long retryInteral, int parallelism) {
        this.dataStream = dataStream;
        this.outJdbcUrl = outJdbcUrl;
        this.maxRetry = maxRetry;
        this.retryInteral = retryInteral;
        this.parallelism = parallelism;
    }

    @Override
    public void run() {
    dataStream
        .filter(new FilterFunction<String>() {
              @Override
              public boolean filter(String value) throws Exception {
                  return JSON.isValid(value);
              }
        })
        .map(new MapFunction<String,JSONObject>() {
            @Override
            public JSONObject map(String value) throws Exception {
                return JSON.parseObject(value);
            }
        })
        .filter(new FilterFunction<JSONObject>() {
              @Override
              public boolean filter(JSONObject jsonObject) throws Exception {
                  Double gmNginxTimestamp = jsonObject.getDouble("gm_nginx_timestamp");
                  if(null != gmNginxTimestamp){
                      long gmNginxTimestampLong = Math.round(gmNginxTimestamp * 1000);
                      String currentDateStr = DateUtils.getCurrentDateStr();
                      long currentDateBegin = DateUtils.getTimestampByDateStr(currentDateStr + " 00:00:00");
                      long currentDateend = DateUtils.getTimestampByDateStr(currentDateStr + " 23:59:59");
                      if(gmNginxTimestampLong >= currentDateBegin && gmNginxTimestampLong <= currentDateend){
                          String type = jsonObject.getString("type");
                          JSONObject paramsObject = jsonObject.getJSONObject("params");
                          JSONObject deviceObject = jsonObject.getJSONObject("device");
                          if(null != paramsObject && null != deviceObject && StringUtils.isNotBlank(type)){
                              String deviceId = deviceObject.getString("device_id");
                              String idfv = deviceObject.getString("idfv");
                              String clId = "";
                              if(StringUtils.isBlank(deviceId) && StringUtils.isNotBlank(idfv)){
                                  clId = idfv;
                              }else{
                                  clId = deviceId;
                              }
                              if(StringUtils.isNotBlank(clId)){
                                  String pageName = paramsObject.getString("page_name");
                                  String tabName = paramsObject.getString("tab_name");
                                  if(null != pageName && null != tabName && "home".equals(pageName.trim()) && "精选".equals(tabName.trim())){
                                      String cardContentType = paramsObject.getString("card_content_type");
                                      if("on_click_post_card".equals(type) || ("on_click_card".equals(type) && "user_post".equals(cardContentType))
                                              || ("search_result_click_infomation_item".equals(type) && "11".equals(paramsObject.getString("business_type")))){
                                          return true;
                                      }
                                      String[] types = {"on_click_topic_card","staggered_topic_click","zone_detail_click_topic","zone_v3_click_diary_topic","diarybook_detail_click_diary_item","on_click_ugc_topic"};
                                      String[] cardContentTypes = {"topic_detail","topic"};
                                      if(Arrays.asList(types).contains(type) || ("on_click_card".equals(type) && Arrays.asList(cardContentTypes).contains(cardContentType))){
                                          return true;
                                      }
                                  }
                                  String referrer = paramsObject.getString("referrer");
                                  if("page_view".equals(type) && "answer_detail".equals(pageName) && "home".equals(referrer)){
                                      return true;
                                  }
                              }
                          }
                      }
                  }
                  return false;
              }
        })
        .map(new MapFunction<JSONObject, CtrEstimateClkEtl>() {
            @Override
            public CtrEstimateClkEtl map(JSONObject jsonObject) throws Exception {
                String type = jsonObject.getString("type");
                JSONObject paramsObject = jsonObject.getJSONObject("params");
                JSONObject deviceObject = jsonObject.getJSONObject("device");
                String deviceId = deviceObject.getString("device_id");
                String idfv = deviceObject.getString("idfv");
                String clId = "";
                if(StringUtils.isBlank(deviceId) && StringUtils.isNotBlank(idfv)){
                    clId = idfv;
                }else{
                    clId = deviceId;
                }
                String cardContentType = paramsObject.getString("card_content_type");
                if("on_click_post_card".equals(type) || ("on_click_card".equals(type) && "user_post".equals(cardContentType))
                        || ("search_result_click_infomation_item".equals(type) && "11".equals(paramsObject.getString("business_type")))){
                    return new CtrEstimateClkEtl(clId,"tractate_card",1);
                }
                String[] types = {"on_click_topic_card","staggered_topic_click","zone_detail_click_topic","zone_v3_click_diary_topic","diarybook_detail_click_diary_item","on_click_ugc_topic"};
                String[] cardContentTypes = {"topic_detail","topic"};
                if(Arrays.asList(types).contains(type) || ("on_click_card".equals(type) && Arrays.asList(cardContentTypes).contains(cardContentType))){
                    return new CtrEstimateClkEtl(clId,"content_card",1);
                }
                String pageName = paramsObject.getString("page_name");
                String referrer = paramsObject.getString("referrer");
                if("page_view".equals(type) && "answer_detail".equals(pageName) && "home".equals(referrer)){
                    return new CtrEstimateClkEtl(clId,"answer_card",1);
                }
                return new CtrEstimateClkEtl();
            }
        })
        .keyBy(new KeySelector<CtrEstimateClkEtl,String>() {
            @Override
            public String getKey(CtrEstimateClkEtl estimateClickEtl) throws Exception {
                return estimateClickEtl.getDeviceId() + "_" + estimateClickEtl.getEstimateType();
            }
        })
        //.timeWindow(Time.minutes(5), Time.minutes(5))
        .timeWindow(Time.seconds(5), Time.seconds(5))
        .process(new ProcessWindowFunction<CtrEstimateClkEtl, DeviceCurrentEstimateClk, String, TimeWindow>() {
            @Override
            public void process(String key, Context context, Iterable<CtrEstimateClkEtl> estimateClickEtls, Collector<DeviceCurrentEstimateClk> out) {
                /* 数据转置
                111	a	1
                111 b   1
                222 a   1
                333 b   1
                device_id   a   b   c
                111			1	0	0
                111			0	1	0
                222			1	0	0
                333			0	1	0 */
                Date date = new Date();
                for (CtrEstimateClkEtl estimateClickEtl : estimateClickEtls) {
                    DeviceCurrentEstimateClk deviceCurrentEstimateClk = new DeviceCurrentEstimateClk();
                    deviceCurrentEstimateClk.setDeviceId(estimateClickEtl.getDeviceId());
                    deviceCurrentEstimateClk.setPartitionDate(DateUtils.getDateStr(date));
                    deviceCurrentEstimateClk.setLastUpdateTime(DateUtils.getTimeStr(date));
                    if("tractate_card".equals(estimateClickEtl.getEstimateType())){
                        deviceCurrentEstimateClk.setTractateCardClick(1L);
                    }else if("content_card".equals(estimateClickEtl.getEstimateType())){
                        deviceCurrentEstimateClk.setContentCardClick(1L);
                    }else if("answer_card".equals(estimateClickEtl.getEstimateType())){
                        deviceCurrentEstimateClk.setAnswerCardClick(1L);
                    }
                    out.collect(deviceCurrentEstimateClk);
                }
            }
        })
        .addSink(new CtrEstimateClkMysqlSink(outJdbcUrl, maxRetry, retryInteral))
        .setParallelism(parallelism);
    }
}
