package com.gmei.data.monitor.operator;

import com.alibaba.fastjson.JSONObject;
import com.gmei.data.monitor.bean.TblMonitorPortraitShd;
import com.gmei.data.monitor.sink.PortraitShdMysqlSink;
import com.gmei.data.monitor.utils.DateUtils;
import org.apache.flink.api.common.functions.FilterFunction;
import org.apache.flink.api.common.functions.MapFunction;
import org.apache.flink.api.java.tuple.Tuple;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.streaming.api.datastream.DataStream;
import org.apache.flink.streaming.api.datastream.SingleOutputStreamOperator;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.streaming.api.functions.windowing.ProcessWindowFunction;
import org.apache.flink.streaming.api.windowing.time.Time;
import org.apache.flink.streaming.api.windowing.windows.TimeWindow;
import org.apache.flink.util.Collector;

import java.util.Arrays;

/**
 * @ClassName PortraitMonitorShdOperator
 * @Description: 用户画像计划打点数监控
 * @Author zhaojianwei
 * @Date 2020/3/18
 * @Version V1.0
 **/
public class PortraitMonitorShdOperator implements BaseOperator{
    private DataStream dataStream;
    private int windownSize;
    private int slideSize;
    private String outJdbcUrl;
    private int maxRetry;
    private long retryInteral;
    private int parallelism;

    public PortraitMonitorShdOperator(DataStream dataStream, int windownSize,int slideSize,String outJdbcUrl, int maxRetry, long retryInteral,int parallelism) {
        this.dataStream = dataStream;
        this.windownSize = windownSize;
        this.slideSize = slideSize;
        this.outJdbcUrl = outJdbcUrl;
        this.maxRetry = maxRetry;
        this.retryInteral = retryInteral;
        this.parallelism = parallelism;
    }

    @Override
    public void run() {
        dataStream
            .map(new MapFunction<String, JSONObject>() {
                @Override
                public JSONObject map(String value) throws Exception {
                    JSONObject jsonObject = new JSONObject();
                    try{
                        jsonObject = JSONObject.parseObject(value);
                    }catch (Exception e){
                        e.printStackTrace();
                    }
                    return jsonObject;
                }
            })
            .filter(new FilterFunction<JSONObject>() {
                @Override
                public boolean filter(JSONObject jsonObject) throws Exception {
                    try {
                        JSONObject sysObject = jsonObject.getJSONObject("SYS");
                        if (null != sysObject) {
                            String action = sysObject.getString("action");
                            if (null != action) {
                                if ("/api/private_conversation/".equals(action)
                                        || "/api/initiate/interest_record".equals(action)
                                        || "/api/one_image/share/v3".equals(action)
                                        || "/gm_ai/face_app/test_skin".equals(action)) {
                                    jsonObject.put("statistics_action", action);
                                    return true;
                                }
                            }
                            JSONObject appObject = sysObject.getJSONObject("APP");
                            if (null != appObject) {
                                String eventType = appObject.getString("event_type");
                                if (null != eventType) {
                                    if ("validate_order".equals(eventType)
                                            || "paid_success".equals(eventType)
                                            || "add_shopcart".equals(eventType)) {
                                        jsonObject.put("statistics_action", eventType);
                                        return true;
                                    }
                                }
                            }
                            String appAction = appObject.getString("action");
                            if (null != appAction) {
                                String[] edits = {"create", "update", "answer"};
                                if (Arrays.asList(edits).contains(appAction)) {
                                    jsonObject.put("statistics_action", appAction);
                                    return true;
                                }
                                String[] interact = {"like", "comment", "collect"};
                                if (Arrays.asList(interact).contains(appAction)) {
                                    jsonObject.put("statistics_action", appAction);
                                    return true;
                                }
                            }
                        }
                        String type = jsonObject.getString("type");
                        String device = jsonObject.getString("device");
                        if (null != type && null != device) {
                            if ("do_search".equals(type)
                                    || "goto_welfare_detail".equals(type)
                                    || "on_click_card".equals(type)
                                    || "home_click_section".equals(type)) {
                                jsonObject.put("statistics_action", type);
                                return true;
                            }
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                        return false;
                    }
                    return false;
                }
            })
            .map(new MapFunction<JSONObject, Tuple2<String, JSONObject>>() {
                @Override
                public Tuple2<String, JSONObject> map(JSONObject jsonObject) throws Exception {
                    String statisticsAction = jsonObject.getString("statistics_action");
                    return new Tuple2<String, JSONObject>(statisticsAction, jsonObject) {
                    };
                }
            })
            .keyBy(0)
            .timeWindow(Time.seconds(windownSize),Time.seconds(slideSize))
            .process(new ProcessWindowFunction<Tuple2<String,JSONObject>, TblMonitorPortraitShd, Tuple, TimeWindow>() {
                @Override
                public void process(Tuple key, Context context, Iterable<Tuple2<String,JSONObject>> elements, Collector<TblMonitorPortraitShd> out) {
                    Integer count = 0;
                    for (Tuple2<String,JSONObject> tuple2 : elements) {
                        ++ count;
                    }
                    String monitorTime = DateUtils.getCurrentTimeStr();
                    TblMonitorPortraitShd tblMonitorPortraitShd = new TblMonitorPortraitShd(key.getField(0), count, monitorTime);
                    out.collect(tblMonitorPortraitShd);
                }
            })
            .addSink(new PortraitShdMysqlSink(outJdbcUrl,maxRetry,retryInteral))
            .setParallelism(parallelism);
    }
}
