package com.gmei.data.monitor;

import com.gmei.data.monitor.operator.PortraitMonitorErrOperator;
import com.gmei.data.monitor.operator.PortraitMonitorShdOperator;
import com.gmei.data.monitor.operator.PortraitMonitorSucOperator;
import com.gmei.data.monitor.source.PortraitKafkaSource;
import com.gmei.data.monitor.source.PortraitSucKafkaSource;
import org.apache.flink.api.common.restartstrategy.RestartStrategies;
import org.apache.flink.api.java.utils.ParameterTool;
import org.apache.flink.runtime.state.filesystem.FsStateBackend;
import org.apache.flink.streaming.api.TimeCharacteristic;
import org.apache.flink.streaming.api.datastream.DataStream;
import org.apache.flink.streaming.api.environment.CheckpointConfig;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;

import java.text.SimpleDateFormat;

/**
 * @ClassName PortraitMonitorMain
 * @Description: 画像打点实时监控主入口
 *
 * @Author zhaojianwei
 * @Date 2020/3/18
 * @Version V1.0
 **/
public class PortraitMonitorMain {

    public static void main(String[] args) throws Exception{
        // 获取运行参数
        ParameterTool parameterTool = ParameterTool.fromArgs(args);
        String inBrokers = parameterTool.get("inBrokers","test003:9092");
        String batchSize = parameterTool.get("batchSize","1000");
        String maidianInTopic = parameterTool.get("maidianInTopic", "test11");
        String backendInTopic = parameterTool.get("backendInTopic","test12");
        String portraitSucInTopic = parameterTool.get("portraitSucInTopic","test13");
        String portraitErrGroupId = parameterTool.get("portraitErrGroupId","flink_monitor_err");
        String portraitShdGroupId = parameterTool.get("portraitShdGroupId","flink_monitor_shd");
        String portraitSucGroupId = parameterTool.get("portraitSucGroupId","flink_monitor_suc");
        Integer windowSize = parameterTool.getInt("windowSize",60000);
        Integer slideSize = parameterTool.getInt("slideSize",3);
        String outJdbcUrl = parameterTool.get("outJdbcUrl",
                "jdbc:mysql://test002:3306/flink_monitor?user=root&password=5OqYM^zLwotJ3oSo&autoReconnect=true&useSSL=false");
        Integer maxRetry = parameterTool.getInt("maxRetry",3);
        Long retryInteral = parameterTool.getLong("retryInteral",3000);
        String checkpointPath = parameterTool.get("checkpointPath","hdfs://bj-gmei-hdfs/user/data/flink/flink-monitor/checkpoint");
        Boolean isStartFromEarliest = parameterTool.getBoolean("isStartFromEarliest",false);
        String startTime = parameterTool.get("startTime");
        Integer parallelism = parameterTool.getInt("parallelism",2);

        // 获得流处理环境对象
        StreamExecutionEnvironment env = StreamExecutionEnvironment.getExecutionEnvironment();
        //env.setStreamTimeCharacteristic(TimeCharacteristic.EventTime);
        env.enableCheckpointing(1000);
        env.setStateBackend(new FsStateBackend(checkpointPath));
        env.setRestartStrategy(RestartStrategies.fixedDelayRestart(1, 3000));
        CheckpointConfig config = env.getCheckpointConfig();
        config.enableExternalizedCheckpoints(CheckpointConfig.ExternalizedCheckpointCleanup.RETAIN_ON_CANCELLATION);

        // 获取数据源
        DataStream portraitErrDataStream = new PortraitKafkaSource(
                env,
                inBrokers,
                maidianInTopic,
                backendInTopic,
                portraitErrGroupId,
                batchSize,
                isStartFromEarliest,
                startTime
        ).getInstance();
        DataStream portraitShdDataStream = new PortraitKafkaSource(
                env,
                inBrokers,
                maidianInTopic,
                backendInTopic,
                portraitShdGroupId,
                batchSize,
                isStartFromEarliest,
                startTime
        ).getInstance();
        DataStream portraitSucDataStream = new PortraitSucKafkaSource(
                env,
                inBrokers,
                portraitSucInTopic,
                portraitSucGroupId,
                batchSize,
                isStartFromEarliest,
                startTime
        ).getInstance();

        // 执行处理核心逻辑
        new PortraitMonitorErrOperator(portraitErrDataStream,outJdbcUrl,maxRetry,retryInteral,parallelism).run();
        new PortraitMonitorShdOperator(portraitShdDataStream,windowSize,slideSize,outJdbcUrl,maxRetry,retryInteral,parallelism).run();
        new PortraitMonitorSucOperator(portraitSucDataStream,windowSize,slideSize,outJdbcUrl,maxRetry,retryInteral,parallelism).run();

        // 常驻执行
        env.execute("Portrait realtime monitor");
    }
}

