package com.gmei.data.monitor.operator;

import com.alibaba.fastjson.JSONObject;
import com.gmei.data.monitor.sink.PortraitErrMysqlSink;
import com.gmei.data.monitor.utils.DateUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.api.common.functions.FilterFunction;
import org.apache.flink.api.common.functions.MapFunction;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.streaming.api.datastream.DataStream;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;

import java.util.Arrays;
import java.util.Date;

/**
 * @ClassName PortraitMonitorErr
 * @Description: 画像打点数据解析异常监控
 * @Author zhaojianwei
 * @Date 2020/3/18
 * @Version V1.0
 **/
public class PortraitMonitorErrOperator implements BaseOperator{
    private DataStream dataStream;
    private String outJdbcUrl;
    private int maxRetry;
    private long retryInteral;
    private int parallelism;
    public static final DateTimeFormatter dateTimeFormat = ISODateTimeFormat.dateTime();
    public static final DateTimeFormatter dateTimeNoMillisFormat = ISODateTimeFormat.dateTimeNoMillis();

    public PortraitMonitorErrOperator(DataStream dataStream, String outJdbcUrl, int maxRetry, long retryInteral,int parallelism) {
        this.dataStream = dataStream;
        this.outJdbcUrl = outJdbcUrl;
        this.maxRetry = maxRetry;
        this.retryInteral = retryInteral;
        this.parallelism = parallelism;
    }

    @Override
    public void run() {
        dataStream
            .map(new MapFunction<String, String>() {
                @Override
                public String map(String value) {
                    try {
                        JSONObject jsonObject = JSONObject.parseObject(value);
                        JSONObject sysObject = jsonObject.getJSONObject("SYS");
                        if (null != sysObject) {
                            String action = sysObject.getString("action");
                            if (null != action) {
                                if ("/api/private_conversation/".equals(action)
                                        || "/api/initiate/interest_record".equals(action)
                                        || "/api/one_image/share/v3".equals(action)
                                        || "/gm_ai/face_app/test_skin".equals(action)) {
                                    jsonObject.put("statistics_action", action);
                                }
                            }
                            JSONObject appObject = sysObject.getJSONObject("APP");
                            if (null != appObject) {
                                String eventType = appObject.getString("event_type");
                                if (null != eventType) {
                                    if ("validate_order".equals(eventType)
                                            || "paid_success".equals(eventType)
                                            || "add_shopcart".equals(eventType)) {
                                        jsonObject.put("statistics_action", eventType);
                                    }
                                }
                                String appAction = appObject.getString("action");
                                if (null != appAction) {
                                    String[] edits = {"create", "update", "answer"};
                                    if (Arrays.asList(edits).contains(appAction)) {
                                        jsonObject.put("statistics_action", appAction);
                                    }
                                    String[] interact = {"like", "comment", "collect"};
                                    if (Arrays.asList(interact).contains(appAction)) {
                                        jsonObject.put("statistics_action", appAction);
                                    }
                                }
                            }
                        }
                        String type = jsonObject.getString("type");
                        String device = jsonObject.getString("device");
                        if (null != type && null != device) {
                            if ("do_search".equals(type)
                                    || "goto_welfare_detail".equals(type)
                                    || "on_click_card".equals(type)
                                    || "home_click_section".equals(type)) {
                                jsonObject.put("statistics_action", type);
                            }
                        }
                        return "";
                    } catch (Exception e) {
                        e.printStackTrace();
                        return value;
                    }
                }
            })
            .filter(new FilterFunction<String>() {
                  @Override
                  public boolean filter(String value) throws Exception {
                      if(StringUtils.isBlank(value)){
                          return false;
                      }
                      return true;
                  }
              })
            .map(new MapFunction<String,Tuple2<String,String>>() {
                @Override
                public Tuple2<String,String> map(String value) throws Exception {
                    long logTime = 0;
                    JSONObject jsonObject = JSONObject.parseObject(value);
                    String maidianEventTime = jsonObject.getString("create_at");
                    if(StringUtils.isNotBlank(maidianEventTime)){
                        logTime = Long.valueOf(maidianEventTime);
                    }
                    String backendEventTime = jsonObject.getString("TIME");
                    if(StringUtils.isNotBlank(backendEventTime)){
                        try {
                            logTime = dateTimeFormat.parseMillis(backendEventTime)/1000;
                        } catch (IllegalArgumentException e) {
                            try {
                                logTime = dateTimeNoMillisFormat.parseMillis(backendEventTime)/1000;
                            } catch (IllegalArgumentException e2) {
                                e2.printStackTrace();
                            }
                        }
                    }
                    String eventTime = DateUtils.secondToDate(logTime);
                    return new Tuple2<>(value,eventTime);
                }
            })
            .addSink(new PortraitErrMysqlSink(outJdbcUrl,maxRetry,retryInteral))
            .setParallelism(parallelism);
    }
}
