package com.gmei.data.monitor.operator;

import com.alibaba.fastjson.JSONObject;
import com.gmei.data.monitor.bean.GmPortraitResult;
import com.gmei.data.monitor.bean.TblMonitorPortraitShd;
import com.gmei.data.monitor.sink.PortraitShdMysqlSink;
import com.gmei.data.monitor.utils.DateUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.api.common.functions.FilterFunction;
import org.apache.flink.api.common.functions.MapFunction;
import org.apache.flink.api.java.tuple.Tuple;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.streaming.api.datastream.DataStream;
import org.apache.flink.streaming.api.datastream.KeyedStream;
import org.apache.flink.streaming.api.datastream.SingleOutputStreamOperator;
import org.apache.flink.streaming.api.datastream.WindowedStream;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.streaming.api.functions.windowing.ProcessWindowFunction;
import org.apache.flink.streaming.api.windowing.time.Time;
import org.apache.flink.streaming.api.windowing.windows.TimeWindow;
import org.apache.flink.util.Collector;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;

import java.util.Arrays;
import java.util.Date;

/**
 * @ClassName PortraitMonitorShdOperator
 * @Description: 用户画像计划打点数监控
 * @Author zhaojianwei
 * @Date 2020/3/18
 * @Version V1.0
 **/
public class PortraitMonitorShdOperator implements BaseOperator{
    private DataStream dataStream;
    private int windownSize;
    private int slideSize;
    private String outJdbcUrl;
    private int maxRetry;
    private long retryInteral;
    private int parallelism;
    public static final DateTimeFormatter dateTimeFormat = ISODateTimeFormat.dateTime();
    public static final DateTimeFormatter dateTimeNoMillisFormat = ISODateTimeFormat.dateTimeNoMillis();

    public PortraitMonitorShdOperator(DataStream dataStream, int windownSize,int slideSize,String outJdbcUrl, int maxRetry, long retryInteral,int parallelism) {
        this.dataStream = dataStream;
        this.windownSize = windownSize;
        this.slideSize = slideSize;
        this.outJdbcUrl = outJdbcUrl;
        this.maxRetry = maxRetry;
        this.retryInteral = retryInteral;
        this.parallelism = parallelism;
    }

    @Override
    public void run() {
        SingleOutputStreamOperator map01 = dataStream.map(new MapFunction<String, JSONObject>() {
            @Override
            public JSONObject map(String value) throws Exception {
                JSONObject jsonObject = new JSONObject();
                try {
                    jsonObject = JSONObject.parseObject(value);
                } catch (Exception e) {
                    e.printStackTrace();
                }
                return jsonObject;
            }
        });
        //map01.print();
        SingleOutputStreamOperator filter01 = map01.filter(new FilterFunction<JSONObject>() {
            @Override
            public boolean filter(JSONObject jsonObject) throws Exception {
                try {
                    JSONObject sysObject = jsonObject.getJSONObject("SYS");
                    if (null != sysObject) {
                        String action = sysObject.getString("action");
                        if (null != action) {
                            if ("/api/private_conversation/".equals(action)
                                    || "/api/initiate/interest_record".equals(action)
                                    || "/api/one_image/share/v3".equals(action)
                                    || "/gm_ai/face_app/test_skin".equals(action)) {
                                jsonObject.put("statistics_action", action);
                                return true;
                            }
                        }
                    }
                    JSONObject appObject = jsonObject.getJSONObject("APP");
                    if (null != appObject) {
                        String eventType = appObject.getString("event_type");
                        if (null != eventType) {
                            if ("validate_order".equals(eventType)
                                    || "paid_success".equals(eventType)
                                    || "add_shopcart".equals(eventType)) {
                                jsonObject.put("statistics_action", eventType);
                                return true;
                            }
                        }
                        String appAction = appObject.getString("action");
                        if (null != appAction) {
                            String[] edits = {"create", "update", "answer"};
                            if (Arrays.asList(edits).contains(appAction)) {
                                jsonObject.put("statistics_action", appAction);
                                return true;
                            }
                            String[] interact = {"like", "comment", "collect"};
                            if (Arrays.asList(interact).contains(appAction)) {
                                jsonObject.put("statistics_action", appAction);
                                return true;
                            }
                        }
                    }
                    String type = jsonObject.getString("type");
                    String device = jsonObject.getString("device");
                    if (null != type && null != device) {
                        if ("do_search".equals(type)
                                || "goto_welfare_detail".equals(type)
                                || "on_click_card".equals(type)
                                || "home_click_section".equals(type)) {
                            jsonObject.put("statistics_action", type);
                            return true;
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    return false;
                }
                return false;
            }
        });
        //filter01.print();
        SingleOutputStreamOperator map02 = filter01.map(new MapFunction<JSONObject, Tuple2<String, JSONObject>>() {
            @Override
            public Tuple2<String, JSONObject> map(JSONObject jsonObject) throws Exception {
                String statisticsAction = jsonObject.getString("statistics_action");
                return new Tuple2<String, JSONObject>(statisticsAction, jsonObject) {
                };
            }
        });
        //map02.print();
        KeyedStream keyedStream = map02.keyBy(0);
        //keyedStream.print();
        WindowedStream windowedStream = keyedStream.timeWindow(Time.seconds(windownSize), Time.seconds(slideSize));
        SingleOutputStreamOperator process = windowedStream.process(new ProcessWindowFunction<Tuple2<String, JSONObject>, TblMonitorPortraitShd, Tuple, TimeWindow>() {
            @Override
            public void process(Tuple key, Context context, Iterable<Tuple2<String, JSONObject>> elements, Collector<TblMonitorPortraitShd> out) {
                Integer count = 0;
                Date date = new Date();
                long currentTimestamp = DateUtils.getCurrentTimestamp(date);
                long tenMinitesAgoTimestamp = DateUtils.getTenMinitesAgoTimestamp(date);
                for (Tuple2<String, JSONObject> tuple2 : elements) {
                    System.err.println(tuple2.f0);
                    long logTime = 0;
                    JSONObject jsonObject = tuple2.f1;
                    String maidianEventTime = jsonObject.getString("create_at");
                    if (StringUtils.isNotBlank(maidianEventTime)) {
                        logTime = Long.valueOf(maidianEventTime);
                    }
                    String backendEventTime = jsonObject.getString("TIME");
                    if (StringUtils.isNotBlank(backendEventTime)) {
                        try {
                            logTime = dateTimeFormat.parseMillis(backendEventTime);
                        } catch (IllegalArgumentException e) {
                            try {
                                logTime = dateTimeNoMillisFormat.parseMillis(backendEventTime);
                            } catch (IllegalArgumentException e2) {
                                e2.printStackTrace();
                            }
                        }
                    }
                    if (logTime >= tenMinitesAgoTimestamp && logTime <= currentTimestamp) {
                        ++count;
                    }
                }
                String currentTimeStr = DateUtils.getCurrentTimeStr(date);
                TblMonitorPortraitShd tblMonitorPortraitShd = new TblMonitorPortraitShd(key.getField(0), count, currentTimeStr);
                out.collect(tblMonitorPortraitShd);
            }
        });
        process.print();
        process.addSink(new PortraitShdMysqlSink(outJdbcUrl,maxRetry,retryInteral))
                .setParallelism(parallelism);
    }
}
