package com.gmei.data.monitor.schama;

import org.apache.flink.api.common.serialization.DeserializationSchema;
import org.apache.flink.api.common.serialization.SerializationSchema;
import org.apache.flink.api.common.typeinfo.BasicTypeInfo;
import org.apache.flink.api.common.typeinfo.TypeInformation;
import org.msgpack.MessagePack;
import org.msgpack.type.Value;
import org.msgpack.type.ValueFactory;

import java.io.IOException;
import java.io.ObjectOutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

import static org.apache.flink.util.Preconditions.checkNotNull;

/**
 * ClassName: GmeiLoggingSchema
 * Reason: backend埋点schama
 * Date: 2020-03-17 00:00:00
 *
 * @author zhaojianwei
 * @since JDK 1.8
 */
public class GmeiLoggingSchema implements DeserializationSchema<String>, SerializationSchema<String> {

    private static final long serialVersionUID = 1L;
    private transient Charset charset;

    public GmeiLoggingSchema() {
        this(StandardCharsets.UTF_8);
    }
    public GmeiLoggingSchema(Charset charset) {
        this.charset = checkNotNull(charset);
    }
    public Charset getCharset() {
        return charset;
    }

    /**
     * 流是否结束
     * @param nextElement
     * @return
     */
    @Override
    public boolean isEndOfStream(String nextElement) {
        return false;
    }

    /**
     * 序列化
     * @param element
     * @return
     */
    @Override
    public byte[] serialize(String element) {
        return element.getBytes(charset);
    }

    /**
     * 获取生产类型
     * @return
     */
    @Override
    public TypeInformation<String> getProducedType() {
        return BasicTypeInfo.STRING_TYPE_INFO;
    }

    /**
     * Kafka Serialization
     * @param message
     * @return
     */
    @Override
    public String deserialize(byte[] message) {
        String contentString = "";
        MessagePack msgpack = new MessagePack();
        Value MSGPACK_STRING_VALUE_CONTENT = ValueFactory.createRawValue("content".getBytes(StandardCharsets.US_ASCII));
        try {
            contentString = msgpack.read(message).asMapValue().get(MSGPACK_STRING_VALUE_CONTENT).asRawValue().getString();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return contentString;
    }


    /**
     * 写入方法
     * @param out
     * @throws IOException
     */
    private void writeObject (ObjectOutputStream out) throws IOException {
        out.defaultWriteObject();
        out.writeUTF(charset.name());
    }

    /**
     * 读取方法
     * @param in
     * @throws IOException
     * @throws ClassNotFoundException
     */
    private void readObject(java.io.ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        String charsetName = in.readUTF();
        this.charset = Charset.forName(charsetName);
    }
}
