package com.gmei.up

import java.util.Properties
import com.typesafe.scalalogging.LazyLogging
import org.apache.flink.api.scala._
import org.apache.flink.streaming.api.scala.{ StreamExecutionEnvironment, DataStream }
import org.apache.flink.streaming.connectors.kafka.{
  FlinkKafkaConsumer,
  FlinkKafkaProducer,
  KafkaDeserializationSchema
}
import org.apache.flink.api.common.typeinfo.{ TypeHint, TypeInformation, Types }
import org.apache.kafka.clients.consumer.ConsumerRecord
import com.alibaba.fastjson.JSON
import com.gmei.up.utils.{ UserInfo, ES, Redis, DingTalk }

class UserInfoDeserializationSchema extends KafkaDeserializationSchema[UserInfo] {
  override def isEndOfStream(t: UserInfo): Boolean = false

  override def deserialize(consumerRecord: ConsumerRecord[Array[Byte], Array[Byte]]): UserInfo = {
    val info = new String(consumerRecord.value(), "UTF-8")
    // println(info)
    val userInfo: UserInfo = JSON.parseObject(info, classOf[UserInfo])
    userInfo
  }

  override def getProducedType: TypeInformation[UserInfo] =
    Types.GENERIC(classOf[UserInfo])
}

object Main extends LazyLogging {
  def main(args: Array[String]): Unit =
    // println("###############")
    // val query = ES.generateServiceDiaryRequest(List("光子嫩肤", "水光针"), List("丰胸", "胸部塑身"), -1)
    // println("###############")
    try {
      logger.info("UserPortrait Flink Started")
      val env = StreamExecutionEnvironment.getExecutionEnvironment

      // TODO read from config
      val kafkaConsumerProperties = new Properties
      kafkaConsumerProperties.setProperty("group.id", "user_portrait_flink_streaming")
      kafkaConsumerProperties.setProperty(
        "bootstrap.servers",
        "172.16.44.25:9092,172.16.44.31:9092,172.16.44.45:9092"
      )

      // TODO remove test
      val kafkaConsumer = new FlinkKafkaConsumer[UserInfo](
        "gm-portrait-update-device-test",
        new UserInfoDeserializationSchema,
        kafkaConsumerProperties
      )

      val stream: DataStream[UserInfo] = env.addSource(kafkaConsumer)

      stream.map { user =>
        try {
          val deviceId = user.deviceId
          val projects = user.projects.toList
          val secondDemands = user.secondDemands.toList
          val secondPositions = user.secondPositions.toList
          val secondSolutions = user.secondSolutions.toList
          val cityId = user.cityId
          val streamTimeBegin: Long = System.currentTimeMillis
          println(deviceId)
          // println(projects.mkString(" "))

          val diaryRead = Redis.getRead(deviceId, "diary")
          val tractateRead = Redis.getRead(deviceId, "tractate")
          val answerRead = Redis.getRead(deviceId, "answer")

          val diaryReq =
            ES.generateDiaryRequest(
              projects,
              secondDemands,
              secondPositions,
              secondSolutions,
              diaryRead,
              cityId,
              300
            )
          val tractateReq =
            ES.generateTractateRequest(
              projects,
              secondDemands,
              secondPositions,
              secondSolutions,
              tractateRead,
              300
            )
          val answerReq =
            ES.generateAnswerRequest(projects, secondDemands, secondPositions, secondSolutions, answerRead, 100)
          val serviceDiaryReq =
            ES.generateServiceDiaryRequest(
              projects,
              secondDemands,
              secondPositions,
              secondSolutions,
              diaryRead,
              cityId,
              100
            )

          Redis.save(ES.request(diaryReq), deviceId, "diary", streamTimeBegin)
          Redis.save(ES.request(tractateReq), deviceId, "tractate", streamTimeBegin)
          Redis.save(ES.request(answerReq), deviceId, "answer", streamTimeBegin)
          Redis.save(ES.request(serviceDiaryReq), deviceId, "service_diary", streamTimeBegin)

          // current time
          logger.info(s"${user.eventCn} ${deviceId}")

          // if (deviceId == "64695DE0-B926-4188-9C62-D987DC20BEDF") {
          //   DingTalk.send(
          //     Map(
          //       "deviceId" -> deviceId,
          //       "eventCn" -> user.eventCn,
          //       "action" -> user.action,
          //       "cityId" -> user.cityId.toString,
          //       "logTime" -> user.logTime.toString,
          //       "projects" -> user.projects.mkString(" "),
          //       "diaryRead" -> diaryRead.size.toString,
          //       "tractateRead" -> tractateRead.size.toString,
          //       "answerRead" -> answerRead.size.toString
          //     )
          //   )
          // }

          deviceId
        } catch {
          case e: Throwable =>
            DingTalk.send(
              Map("Main.main" -> "stream.map", "error" -> e.getStackTrace.mkString("\n")),
              contentType = "exception"
            )
            logger.error(e.getStackTrace.mkString("\n"))
            e.printStackTrace()
            ""
        }
      }

      // stream.print
      env.execute("flink streaming user portrait")

    } catch {
      case e: Throwable =>
        DingTalk.send(
          Map("method" -> "main", "error" -> e.getStackTrace.mkString("\n")),
          contentType = "exception"
        )
        logger.error(e.getStackTrace.mkString("\n"))
        e.printStackTrace()
    }
}
