#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging
import traceback
from libs.es import ESPerform


class GroupUtils(object):

    @classmethod
    def get_group_query_result(cls, query, offset, size):
        try:
            q = dict()

            multi_fields = {
                'description': 2,
                'name': 4,
            }
            query_fields = ['^'.join((k, str(v))) for (k, v) in multi_fields.items()]
            multi_match = {
                'query': query,
                'type': 'cross_fields',
                'operator': 'and',
                'fields': query_fields,
            }

            q['query'] = {
                'bool': {
                    "must": [
                        {"term": {"is_online": True}},
                        {"term": {"is_deleted": False}}
                    ],
                    "should": [
                        {'multi_match': multi_match}
                    ],
                    "minimum_should_match": 1
                }
            }
            q["_source"] = {
                "includes": ["id"]
            }

            return ESPerform.get_search_results(ESPerform.get_cli(), "group", q, offset, size)
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return {"total_count": 0, "hits": []}

    @classmethod
    def get_hot_pictorial_recommend_result_list(cls, offset, size, es_cli_obj=None):

        try:
            if not es_cli_obj:
                es_cli_obj = ESPerform.get_cli()

            q = dict()
            q["query"] = {
                "bool": {
                    "must": [
                        {"term": {"is_online": True}},
                        {"term": {"is_deleted": False}}
                    ]
                }
            }
            q["sort"] = [
                {"high_quality_topic_num": {"order": "desc"}}
            ]
            q["_source"] = {
                "includes": ["id"]
            }
            MAX_LOAD = 100
            sizes = max(size, MAX_LOAD)
            result_dict = ESPerform.get_search_results(es_cli_obj, "pictorial", q, offset, sizes)

            pictorial_ids_list = []
            if len(result_dict["hits"]) > 0:
                pictorial_ids_list = [item["_source"]["id"] for item in result_dict["hits"]]

            pictorial_del_ids_list = []
            if len(pictorial_ids_list) > 0:
                for id in pictorial_ids_list:
                    q ={
                        "query": {
                            "bool": {
                                "must": [{
                                    "term": {
                                        "pictorial_id": id
                                    }
                                }, {
                                    "terms": {
                                        "content_level": [3, 4, 5,0]
                                    }
                                }, {
                                    "term": {
                                        "is_online": True
                                    }
                                }]
                            }
                        }
                    }
                    topic_ids_list = []
                    result_dict = ESPerform.get_search_results(es_cli_obj, "topic", q, offset, size)

                    if len(result_dict["hits"]) > 0:
                        topic_ids_list = [item["_source"]["id"] for item in result_dict["hits"]]

                    if len(topic_ids_list) > 5:
                        pictorial_del_ids_list.append(id)

                return pictorial_del_ids_list

            else:
                return []
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []


    @classmethod
    def get_user_attention_pictorial_list(cls, user_id, offset=0, size=10, es_cli_obj=None):
        """
        :remark: 获取用户关注小组列表
        :return:
        """
        try:
            if not es_cli_obj:
                es_cli_obj = ESPerform.get_cli()

            q = dict()
            q["query"] = {
                "bool": {
                    "must": [
                        {"term": {"is_online": True}},
                        {"term": {"user_id": user_id}},
                        {"term": {"is_deleted": False}}
                    ]
                }
            }
            q["_source"] = {
                "includes": ["attention_pictorial_id_list"]
            }
            MAX_LOAD = 100
            sizes = max(size, MAX_LOAD)

            result_dict = ESPerform.get_search_results(es_cli_obj, "user", q, offset, sizes)
            pictorial_ids_list = []
            if len(result_dict["hits"]) > 0:
                pictorial_ids_list = result_dict["hits"][0]["_source"]["attention_pictorial_id_list"]

            pictorial_del_ids_list = []
            if len(pictorial_ids_list) > 0:
                for id in pictorial_ids_list:
                    q = {
                        "query": {
                            "bool": {
                                "must": [{
                                    "term": {
                                        "pictorial_id": id
                                    }
                                }, {
                                    "terms": {
                                        "content_level": [3, 4, 5,0]
                                    }
                                }, {
                                    "term": {
                                        "is_online": True
                                    }
                                }]
                            }
                        }
                    }
                    topic_ids_list = []
                    result_dict = ESPerform.get_search_results(es_cli_obj, "topic", q, offset, size)

                    if len(result_dict["hits"]) > 0:
                        topic_ids_list = [item["_source"]["id"] for item in result_dict["hits"]]

                    if len(topic_ids_list) > 5:
                        pictorial_del_ids_list.append(id)

                return pictorial_del_ids_list

            else:
                return []

        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []


    @classmethod
    def get_pictorial_ids_by_aggs(cls, pictorial_ids_list, es_cli_obj=None):
        """
        :remark:聚合查询获取小组列表
        :param group_id_list:
        :return:
        """
        try:
            if not es_cli_obj:
                es_cli_obj = ESPerform.get_cli()

            q = dict()
            q["size"] = 0
            q["query"] = {
                "terms": {
                    "pictorial_id": pictorial_ids_list
                }
            }
            q["aggs"] = {
                "pictorial_ids": {
                    "terms": {
                        "field": "pictorial_id"
                    },
                    "aggs": {
                        "max_date": {
                            "max": {
                                "field": "update_time_val"
                            }
                        }
                    }
                }
            }

            result_dict = ESPerform.get_search_results(es_cli_obj, "topic", q, aggregations_query=True)
            buckets_list = result_dict["aggregations"]["pictorial_ids"]["buckets"]

            sorted_buckets_list = sorted(buckets_list, key=lambda item: item["max_date"]["value"], reverse=True)
            sorted_pictorial_id_list = [item["key"] for item in sorted_buckets_list]

            return sorted_pictorial_id_list
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []
