import datetime
from django.db import models
import logging
import traceback
from django.conf import settings
from trans2es.models.topic import Reply
from .tag import Tag
from .topic import Topic, TopicBillBoard
from .user_extra import UserExtra


class PictorialFollow(models.Model):
    """画报关注"""

    class Meta:
        verbose_name = u"画报用户关系"
        app_label = "community"
        db_table = "community_pictorial_follow"

    id = models.IntegerField(verbose_name=u'关注ID', primary_key=True)
    create_time = models.DateTimeField(verbose_name=u'创建时间', default=datetime.datetime.fromtimestamp(0))
    update_time = models.DateTimeField(verbose_name=u'更新时间', default=datetime.datetime.fromtimestamp(0))
    is_online = models.BooleanField(verbose_name=u'是否上线')
    is_deleted = models.BooleanField(verbose_name=u'是否删除')
    pictorial_id = models.BigIntegerField(verbose_name=u'画报ID')
    user_id = models.BigIntegerField(verbose_name=u'用户ID')


class PictorialTopics(models.Model):
    """画报帖子关系"""

    class Meta:
        verbose_name = u'画报帖子关系'
        app_label = 'community'
        db_table = 'community_pictorial_topic'

    id = models.IntegerField(verbose_name=u'日记ID', primary_key=True)
    pictorial_id = models.BigIntegerField(verbose_name=u'画报ID')
    topic_id = models.BigIntegerField(verbose_name=u'帖子ID')
    is_online = models.BooleanField(verbose_name=u"是否有效", default=True)
    is_deleted = models.BooleanField(verbose_name=u'是否删除')


class Pictorial(models.Model):
    """画报关注"""

    class Meta:
        verbose_name = u"画报"
        app_label = "community"
        db_table = "community_pictorial"

    id = models.IntegerField(verbose_name=u'关注ID', primary_key=True)
    create_time = models.DateTimeField(verbose_name=u'创建时间', default=datetime.datetime.fromtimestamp(0))
    update_time = models.DateTimeField(verbose_name=u'更新时间', default=datetime.datetime.fromtimestamp(0))
    is_online = models.BooleanField(verbose_name=u'是否上线')
    is_deleted = models.BooleanField(verbose_name=u'是否删除')
    is_recommend = models.BooleanField(verbose_name=u'推荐')
    name = models.CharField(verbose_name=u'画报名称', max_length=100)
    description = models.CharField(verbose_name=u'画报描述', max_length=200)
    creator_id = models.BigIntegerField(verbose_name=u'画报用户ID')
    icon = models.CharField(verbose_name=u'画报名称', max_length=255)
    topic_num = models.IntegerField(verbose_name=u'次数')
    add_score = models.IntegerField(verbose_name=u'人工权重')
    is_default = models.IntegerField(verbose_name=u"是否是默认画报")

    def get_topic_id(self):
        try:
            topic_id_list = list(
                PictorialTopics.objects.filter(pictorial_id=self.id, is_online=True, is_deleted=False).values_list(
                    "topic_id", flat=True))
            return topic_id_list
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []

    def get_effective(self, topic_id_list):
        try:
            effective_num = 0
            ret = False

            for topic_id in topic_id_list:
                topic_id_object = Topic.objects.filter(id=int(topic_id)).first()
                if topic_id_object and topic_id_object.is_online and int(topic_id_object.content_level) in [0, 3, 4, 5,
                                                                                                            6]:
                    effective_num += 1
                    if effective_num >= 5:
                        ret = True
                        break

            return ret
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return False

    # 获取前一天4，5星发帖数
    def get_high_quality_topic_num(self):
        try:
            yesterday = datetime.datetime.now() - datetime.timedelta(days=1)
            yesterday_begin_time = "%s-%s-%s 00:00:00" % (yesterday.year, yesterday.month, yesterday.day)
            yesterday_end_time = "%s-%s-%s 23:59:59" % (yesterday.year, yesterday.month, yesterday.day)

            topic_id_list = PictorialTopics.objects.filter(pictorial_id=self.id).values_list("topic_id", flat=True)

            topic_num = Topic.objects.filter(content_level__in=["4", "5"], create_time__gte=yesterday_begin_time,
                                             create_time__lte=yesterday_end_time, id__in=topic_id_list).count()

            return topic_num

        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return False

    def get_tag_by_id(self):
        try:
            tag_id_list = list(
                PictorialTag.objects.filter(pictorial_id=self.id, is_online=True).values_list("tag_id", flat=True))

            return tag_id_list
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []

    def get_tag_by_name(self, tag_id):
        try:
            tag_name_list = list(Tag.objects.filter(id__in=tag_id, is_online=True).values_list("name", flat=True))

            return tag_name_list
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []

    def get_edit_tag_id(self):
        try:
            tag_id_list = list(
                PictorialTag.objects.filter(pictorial_id=self.id, is_online=True, is_collection=1).values_list("tag_id",
                                                                                                               flat=True))

            return tag_id_list
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []

    def get_edit_tag_name(self, tag_id):
        try:
            tag_name_list = list(
                Tag.objects.filter(id__in=tag_id, is_online=True).values_list("name", flat=True))

            return tag_name_list
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []

    def get_is_cover(self, topic_id_list):
        try:
            effective_num = 0
            ret = False

            for topic_id in topic_id_list:
                topic_id_object = Topic.objects.filter(id=int(topic_id)).first()
                if topic_id_object and topic_id_object.is_online and int(topic_id_object.content_level) in [3, 4, 5, 6]:
                    effective_num += 1
                    if effective_num >= 5:
                        ret = True
                        break

            return ret

        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return False

    def get_topic_vote_number(self):
        try:
            topic_vote = TopicBillBoard.objects.filter(pictorial_id=self.id).values("real_vote_cnt", "virt_vote_cnt")
            total_vote_cnt = 0
            if topic_vote:
                for item in topic_vote:
                    total_vote_cnt += int(item["virt_vote_cnt"]) + int(item["real_vote_cnt"])

            return total_vote_cnt

        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return 0

    def get_activity_join(self):
        try:
            activicy_list = list(CommunityPictorialActivityRelation.objects.filter(pictorial_id=self.id, is_online=True,
                                                                                   is_deleted=False).values_list(
                "pictorial_activity_id", flat=True))

            return activicy_list
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []

    def get_latest_real_reply_time(self):
        try:
            reply_query_results = Reply.objects.using(settings.SLAVE_DB_NAME).filter(pictorial_id=self.id, is_deleted=0,
                                                                                     is_online=1).values("user_id",
                                                                                                         "create_time")
            datetime_list = list()
            if reply_query_results.count() > 0:
                for reply in reply_query_results:
                    UserExtra_results = UserExtra.objects.using(settings.SLAVE_DB_NAME).filter(
                        user_id=reply['user_id']).values('is_shadow')
                    if UserExtra_results.count() > 0:
                        if not UserExtra_results[0]['is_shadow']:
                            datetime_list.append(reply['create_time'])
                    else:
                        datetime_list.append(reply['create_time'])
            if len(datetime_list) > 0:
                dt = max(datetime_list)
                return datetime.datetime(dt.year, dt.month, dt.day, dt.hour, dt.minute, dt.second,
                                         tzinfo=datetime.timezone(datetime.timedelta(hours=8)))
            else:
                return datetime.datetime(1980, 1, 1, 0, 0, 0, tzinfo=datetime.timezone(datetime.timedelta(hours=8)))
        except:
            return datetime.datetime(1980, 1, 1, 0, 0, 0, tzinfo=datetime.timezone(datetime.timedelta(hours=8)))

    def get_latest_real_topic_time(self):
        try:
            topic_id_list = list(
                PictorialTopics.objects.filter(pictorial_id=self.id, is_online=True, is_deleted=False).values_list(
                    "topic_id", flat=True))

            logging.info("get  topic_id_list:%s" % topic_id_list)
            topic_info = Topic.objects.filter(id__in=topic_id_list).values("user_id", "create_time")
            logging.info("get  topic_info:%s" % topic_info)
            datetime_list = list()
            if topic_info.count() > 0:
                for topic in topic_info:
                    UserExtra_results = UserExtra.objects.using(settings.SLAVE_DB_NAME).filter(
                        user_id=topic['user_id']).values('is_shadow')
                    if UserExtra_results.count() > 0:
                        if not UserExtra_results[0]['is_shadow']:
                            datetime_list.append(topic['create_time'])
                    else:
                        datetime_list.append(topic['create_time'])

            logging.info("get  datetime_list:%s" % datetime_list)

            if len(datetime_list) > 0:
                dt = max(datetime_list)
                logging.info("get  max  datetime_list:%s" % dt)

                return datetime.datetime(dt.year, dt.month, dt.day, dt.hour, dt.minute, dt.second,
                                         tzinfo=datetime.timezone(datetime.timedelta(hours=8)))
            else:
                return datetime.datetime(1980, 1, 1, 0, 0, 0, tzinfo=datetime.timezone(datetime.timedelta(hours=8)))
        except:
            return datetime.datetime(1980, 1, 1, 0, 0, 0, tzinfo=datetime.timezone(datetime.timedelta(hours=8)))

    def get_real_user_activate_time(self):
        try:
            reply_query_results = Reply.objects.using(settings.SLAVE_DB_NAME).filter(pictorial_id=self.id, is_deleted=0,
                                                                                     is_online=1).values("user_id",
                                                                                                         "create_time")
            datetime_list = list()
            if reply_query_results.count() > 0:
                for reply in reply_query_results:
                    UserExtra_results = UserExtra.objects.using(settings.SLAVE_DB_NAME).filter(
                        user_id=reply['user_id']).values('is_shadow')
                    if UserExtra_results.count() > 0:
                        if not UserExtra_results[0]['is_shadow']:
                            datetime_list.append(reply['create_time'])
                    else:
                        datetime_list.append(reply['create_time'])

            topic_id_list = list(
                PictorialTopics.objects.filter(pictorial_id=self.id, is_online=True, is_deleted=False).values_list(
                    "topic_id", flat=True))

            topic_info = Topic.objects.filter(id__in=topic_id_list, is_online=1).values("user_id", "create_time")

            if topic_info.count() > 0:
                for topic in topic_info:
                    UserExtra_results = UserExtra.objects.using(settings.SLAVE_DB_NAME).filter(
                        user_id=topic['user_id']).values('is_shadow')
                    if UserExtra_results.count() > 0:
                        if not UserExtra_results[0]['is_shadow']:
                            datetime_list.append(topic['create_time'])
                    else:
                        datetime_list.append(topic['create_time'])

            logging.info("get  datetime_list:%s" % datetime_list)

            if len(datetime_list) > 0:
                dt = max(datetime_list)
                logging.info("get  max  datetime_list:%s" % dt)
                return datetime.datetime(dt.year, dt.month, dt.day, dt.hour, dt.minute, dt.second,
                                         tzinfo=datetime.timezone(datetime.timedelta(hours=8)))

            else:
                return datetime.datetime(1980, 1, 1, 0, 0, 0, tzinfo=datetime.timezone(datetime.timedelta(hours=8)))
        except:
            return datetime.datetime(1980, 1, 1, 0, 0, 0, tzinfo=datetime.timezone(datetime.timedelta(hours=8)))


class PictorialTag(models.Model):
    """画报关注标签"""

    class Meta:
        verbose_name = u"画报标签"
        app_label = "community"
        db_table = "community_pictorial_tag"

    id = models.IntegerField(verbose_name=u'关注ID', primary_key=True)
    create_time = models.DateTimeField(verbose_name=u'创建时间', default=datetime.datetime.fromtimestamp(0))
    update_time = models.DateTimeField(verbose_name=u'更新时间', default=datetime.datetime.fromtimestamp(0))
    pictorial_id = models.BigIntegerField(verbose_name=u'画报ID', max_length=20)
    tag_id = models.BigIntegerField(verbose_name=u'标签ID', max_length=20)
    is_online = models.BooleanField(verbose_name=u'是否上线', max_length=1)
    is_collection = models.IntegerField(verbose_name=u"是否编辑标签")


class CommunityPictorialActivity(models.Model):
    """榜单活动"""

    class Meta:
        verbose_name = u"画报标签"
        app_label = "community"
        db_table = "community_pictorial_activity"

    start_time = models.DateTimeField(verbose_name=u'开始时间', default=datetime.datetime.fromtimestamp(0))
    end_time = models.DateTimeField(verbose_name=u'结束时间', default=datetime.datetime.fromtimestamp(0))
    is_online = models.BooleanField(verbose_name=u'是否上线', max_length=1)
    is_deleted = models.BooleanField(verbose_name=u'是否被删除', max_length=1)


class CommunityPictorialActivityRelation(models.Model):
    """榜单活动关系"""

    class Meta:
        verbose_name = u"画报标签"
        app_label = "community"
        db_table = "community_pictorial_activity_relation"

    is_online = models.BooleanField(verbose_name=u'是否上线', max_length=1)
    is_deleted = models.BooleanField(verbose_name=u'是否被删除', max_length=1)
    pictorial_activity_id = models.BigIntegerField(verbose_name=u'活动ID', max_length=20)
    pictorial_id = models.BigIntegerField(verbose_name=u'榜单ID', max_length=20)


class CommunityPictorialHomeFeed(models.Model):
    """运营榜单"""

    class Meta:
        verbose_name = u"运营画报"
        app_label = "community"
        db_table = "community_pictorial_home_feed"
    is_online = models.BooleanField(verbose_name=u'是否上线', max_length=1)
    is_deleted = models.BooleanField(verbose_name=u'是否被删除', max_length=1)
    pictorial_id = models.BigIntegerField(verbose_name=u'榜单ID', max_length=20)
