
import functools
import logging

from django.conf import settings
from django.core.wsgi import WSGIHandler
from django.core.cache import cache
from django.db import signals

from gm_rpcd import all
from engine.middleware.rpc import HeliosController
from engine.rpc_base import rpc_invoker

_all__ = ['rpc_invoker', 'bind', 'Bind', 'get_current_rpc_invoker']


def bind(endpoint, **options):

    endpoint = 'saturn/' + endpoint
    return all.bind(endpoint, **options)


def get_current_rpc_invoker():
    return HeliosController.get_current_rpc_invoker()


class Context(object):

    @property
    def user(self):
        key = 'session:{}'.format(all.context.session_id)
        user_info = cache.get(key)
        if not user_info:
            user_info = all.context.rpc['passport/info/by_session']().unwrap()
            cache.set(key, user_info, timeout=180)
        return user_info

    @property
    def rpc(self):
        return all.context.rpc


class Bind(object):
    project = 'saturn'
    app = ''
    exception_map = {}

    def _bind(self, endpoint, with_context=False):
        def _wrap(func):
            @functools.wraps(func)
            def _wrapper(*args, **kwargs):
                self.request_start(*args, **kwargs)
                try:
                    if with_context:
                        data = func(Context(), *args, **kwargs)
                    else:
                        data = func(*args, **kwargs)
                except Exception as e:
                    raise self.exception_map.get(type(e), e)
                finally:
                    self.request_end(*args, **kwargs)

                return self.format_data(data)

            return all.bind('{}/{}/{}'.format(self.project, self.app, endpoint))(_wrapper)

        return _wrap

    def request_start(self, *args, **kwargs):
        signals.request_started.send(sender=WSGIHandler)

    def request_end(self, *args, **kwargs):
        signals.request_finished.send(sender=WSGIHandler)

    def format_data(self, data):
        return data

    def bind_context(self, endpoint):
        return self._bind(endpoint, with_context=True)

    def __call__(self, endpoint):
        return self._bind(endpoint)
